#
# TWiki WikiClone (see $wikiversion for version)
#
# Configuration and custom extensions for wiki.pm of TWiki.
#
# Copyright (C) 1999, 2000 Peter Thoeny, TakeFive Software Inc., 
# peter.thoeny@takefive.com , peter.thoeny@attglobal.net
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details, published at 
# http://www.gnu.ai.mit.edu/copyleft/gpl.html 
#
# Notes:
# - Latest version at http://your.domain.com/twiki
# - Installation instructions in $dataDir/TWiki/TWikiDocumentation.txt
# - Customize variables in wikicfg.pm when installing TWiki.
#   NOTE: Don't forget to customize also the Main.TWikiPreferences topic.
# - Optionally change wikicfg.pm for custom extensions of rendering rules.
# - Files wikifcg.pm and wikisearch.pm are included by wiki.pm
# - Upgrading TWiki is easy as long as you do not customize wiki.pm.
# - Variables that can be accessed from topics (see details in
#   TWikiDocumentation.html) :
#       %TOPIC%          name of current topic
#       %WEB%            name of current web
#       %WIKIHOMEURL%    link of top left icon
#       %SCRIPTURL%      base TWiki script URL (place of view, edit...)
#       %SCRIPTURLPATH%  like %SCRIPTURL%, but path only (cut protocol and domain)
#       %SCRIPTSUFFIX%   script suffix (empty by default, ".pl" if required)
#       %PUBURL%         public URL (root of attachment URL)
#       %PUBURLPATH%     path of public URL
#       %ATTACHURL%      attachment URL of current topic
#       %ATTACHURLPATH%  path of attachment URL of current topic
#       %DATE%           today's date
#       %WIKIVERSION%    tool version
#       %USERNAME%       login user name
#       %WIKIUSERNAME%   wiki user name
#       %WIKITOOLNAME%   tool name (TWiki)
#       %MAINWEB%        main web name (Main)
#       %TWIKIWEB%       TWiki system web name (TWiki)
#       %HOMETOPIC%      home topic name (WebHome)
#       %NOTIFYTOPIC%    notify topic name (WebNotify)
#       %WIKIUSERSTOPIC% user list topic name (TWikiUsers)
#       %WIKIPREFSTOPIC% site-level preferences topic name (TWikiPreferences)
#       %WEBPREFSTOPIC%  web preferences topic name (WebPreferences)
#       %STATISTICSTOPIC statistics topic name (WebStatistics)
#       %INCLUDE{...}%   server side include
#       %SEARCH{...}%    inline search


# variables that need to be changed when installing on a new server:
# ==================================================================
#                   %WIKIHOMEURL% : link of TWiki icon in upper left corner :
$wikiHomeUrl      = "http://192.168.222.41/twiki";
#                   Host of TWiki URL :    (Example "http://myhost.com:123")
$defaultUrlHost   = "http://192.168.222.41";
#                   %SCRIPTURLPATH% : cgi-bin path of TWiki URL:
$scriptUrlPath    = "/twiki/bin";
#                   %PUBURLPATH% : Public data path of TWiki URL (root of attachments) :
$pubUrlPath       = "/twiki/pub";
#                   Public data directory, must match $pubUrlPath :
$pubDir           = "/usr/local/twiki/pub";
#                   Template directory :
$templateDir      = "/usr/local/twiki/templates";
#                   Data (topic files) root directory :
$dataDir          = "/usr/local/twiki/data";

# variables that might need to be changed:
# ==================================================================
#                   %SCRIPTSUFFIX% : Suffix of TWiki Perl scripts (i.e. ".pl") :
$scriptSuffix     = "";
#                   set ENV{'PATH'} explicitly for taint checks ( #!perl -T option ) :
$envPath          = "/bin:/usr/bin";
#                   mail program :
$mailProgram      = "/usr/sbin/sendmail -t -oi -oeq";
#                   RCS directory (find out by 'which rcs') :
$rcsDir           = "/usr/bin";
#                   RCS check in command :
$revCiCmd         = "$rcsDir/ci -l -q -mnone -t-none -w'%USERNAME%' %FILENAME%";
#                   RCS check in command with date :
$revCiDateCmd     = "$rcsDir/ci -l -q -mnone -t-none -d'%DATE%' -w'%USERNAME%' %FILENAME%";
#                   RCS check out command :
$revCoCmd         = "$rcsDir/co -q -p%REVISION% %FILENAME%";
#                   RCS history command :
$revHistCmd       = "$rcsDir/rlog -h %FILENAME%";
#                   RCS history on revision command :
$revInfoCmd       = "$rcsDir/rlog -r%REVISION% %FILENAME%";
#                   RCS revision diff command :
$revDiffCmd       = "$rcsDir/rcsdiff -q -w -B -r%REVISION1% -r%REVISION2% %FILENAME%";
#                   RCS delete revision command :
$revDelRevCmd     = "$rcsDir/rcs -q -o%REVISION% %FILENAME%";
#                   RCS unlock command :
$revUnlockCmd     = "$rcsDir/rcs -q -u %FILENAME%";
#                   RCS lock command :
$revLockCmd       = "$rcsDir/rcs -q -l %FILENAME%";
#                   Unix ls command :
$lsCmd            = "/bin/ls";
#                   Unix cp command :
$cpCmd            = "/bin/cp";
#                   Unix egrep command :
$egrepCmd         = "/bin/egrep";
#                   Unix fgrep command :
$fgrepCmd         = "/bin/fgrep";

# variables that probably do not change:
# ==================================================================
#                   %WIKITOOLNAME% : TWiki tool name, default "TWiki" :
$wikiToolName       = "NuTWiki";
#                   Regex security filter for web name, topic name, user name :
$securityFilter     = "[\\\*\?\~\^\$\@\%\`\"\'\&\;\|\<\>\x00-\x1F]";
#                   Default user name, default "guest" :
$defaultUserName    = "guest";
#                   %MAINWEB% : Name of Main web, default "Main" :
$mainWebname        = "Main";
#                   %TWIKIWEB% : Name of TWiki system web, default "TWiki" :
$twikiWebname       = "TWiki";
#                   Pathname of debug file :
$debugFilename      = "$dataDir/debug.txt";
#                   Pathname of user name/password file for authentication :
$htpasswdFilename   = "$dataDir/.htpasswd";
#                   Pathname of log file :
$logFilename        = "$dataDir/log%DATE%.txt";
#                   %WIKIUSERSTOPIC% : Name of users list topic :
$wikiUsersTopicname = "TWikiUsers";
#                   Pathname of users topic, used to translate Intranet name to Wiki name :
$userListFilename   = "$dataDir/$mainWebname/$wikiUsersTopicname.txt";
#                   %HOMETOPIC% : Name of main topic in a web, default "WebHome" :
$mainTopicname      = "WebHome";
#                   %NOTIFYTOPIC% : Name of topic for email notifications, default "WebNotify" :
$notifyTopicname  = "WebNotify";
#                   %WIKIPREFSTOPIC% : Name of site-level preferences topic, default "TWikiPreferences" :
$wikiPrefsTopicname = "TWikiPreferences";
#                   %WEBPREFSTOPIC% : Name of preferences topic in a web, default "WebPreferences" :
$webPrefsTopicname  = "WebPreferences";
#                   %STATISTICSTOPIC% : Name of statistics topic, default "WebStatistics" :
$statisticsTopicname = "WebStatistics";
#                   Number of top viewed topics to show in statistics topic, default "10" :
$statsTopViews      = "10";
#                   Number of top contributors to show in statistics topic, default "10" :
$statsTopContrib    = "10";
#                   Show how many number of revision links, "0" for all, default "4" :
$numberOfRevisions  = "4";
#                   Number of seconds a topic is locked during edit, default "3600" :
$editLockTime       = "3600";

# flag variables that could change:
# ==================================================================
# values are "0" for no, or "1" for yes
#                   Keep same revision if topic is saved again within edit lock time. Default "1"
$doKeepRevIfEditLock = "1";
#                   Remove port number from URL. Default "0"
$doRemovePortNumber = "0";
#                   Change non existing plural topic name to singular,
#                   e.g. TestPolicies to TestPolicy. Default "1"
$doPluralToSingular = "1";
#                   Remove ".." from %INCLUDE{""}% filename, to
#                   prevent includes of "../../file". Default "1"
$doSecureInclude    = "1";
#                   Log topic views to $logFilename. Default "0"
$doLogTopicView     = "0";
#                   Log topic saves to $logFilename. Default "0"
$doLogTopicEdit     = "0";
#                   Log topic saves to $logFilename. Default "1"
$doLogTopicSave     = "1";
#                   Log view attach to $logFilename. Default "0"
$doLogTopicAttach   = "0";
#                   Log file upload to $logFilename. Default "1"
$doLogTopicUpload   = "1";
#                   Log topic rdiffs to $logFilename. Default "0"
$doLogTopicRdiff    = "0";
#                   Log view changes to $logFilename. Default "0"
$doLogTopicChanges  = "0";
#                   Log view changes to $logFilename. Default "0"
$doLogTopicSearch   = "0";
#                   Log user registration to $logFilename. Default "1"
$doLogRegistration  = "1";

# NOTE: Don't forget to customize also the TWiki.TWikiPreferences topic.


# =========================
sub extendHandleCommonTags
{
    # This is the place to define customized tags and variables
    # Called by sub handleCommonTags, after %INCLUDE:"..."%

    my( $text, $topic, $theWeb ) = @_;

    # for compatibility for earlier TWiki versions:
    $text=~ s/%INCLUDE:"(.*?)"%/&handleIncludeFile($1)/geo;
    $text=~ s/%INCLUDE:"(.*?)"%/&handleIncludeFile($1)/geo;  # allow two level includes

    # do custom extension rule, like for example:
    $text=~ s/%WIKIWEB%/$wikiToolName.$theWeb/go;

    # ========================== START XP TAGS ========================== RJB 2001.03.13

    # %XPSHOWSTORIES% - Show all stories
    $text =~ s/%XPSHOWSTORIES%/&xpShowStories($theWeb)/geo;

    # %XPSHOWITERATION% - Show iteration status
    $text =~ s/%XPSHOWITERATION\{(.*?)\}%/&xpShowIteration($1,$theWeb)/geo;

    # %XPSHOWPROJECTSTORIES% - Show project status by stories
    $text =~ s/%XPSHOWPROJECTSTORIES%/&xpShowProjectStories($theWeb)/geo;

    # %XPSHOWPROJECTTASKS% - Show project status by tasks
    $text =~ s/%XPSHOWPROJECTTASKS%/&xpShowProjectTasks($theWeb)/geo;

    # %XPVELOCITIES% - Show velocities by iteration
    $text =~ s/%XPVELOCITIES\{(.*?)\}%/&xpShowVelocities($1,$theWeb)/geo;

    # %XPDUMPITERATION% - Dumps an iteration for printing
    $text =~ s/%XPDUMPITERATION\{(.*?)\}%/&xpDumpIteration($1,$theWeb)/geo;

    # %XPTEST% - Testing
    $text =~ s/%XPTEST\{(.*?)\}%/&xpTest($1,$theWeb)/geo;

    # ========================== END XP TAGS ==========================

    return $text;
}


# =========================
sub extendGetRenderedVersionOutsidePRE
{
    # This is the place to define customized rendering rules
    # Called by sub getRenderedVersion, in loop outside of <PRE> tag

    my( $text, $theWeb ) = @_;

    # do custom extension rule, like for example:
    # s/old/new/go;

    # render *_text_* as "bold italic" text:
    s/(^|\s)\*_([^\s].*?[^\s])_\*(\s|$)/$1<STRONG><EM>$2<\/EM><\/STRONG>$3/go;

    # Use alternate %Web:WikiName% syntax (versus the standard Web.WikiName).
    # This is an old JosWiki render option. (Uncomment for JosWiki compatibility)
#    s/(^|\s|\()\%([^\s].*?[^\s]):([^\s].*?[^\s])\%/&internalLink($2,$3,"$2:$3",$1,1)/geo;

    # Use "forced" non-WikiName links (i.e. %Linkname%)
    # This is an old JosWiki render option. (Uncomment for JosWiki compatibility)
#    s/(^|\s|\()\%([^\s].*?[^\s])\%/&internalLink($theWeb,$2,$2,$1,1)/geo;

    # Use "forced" non-WikiName links (i.e. %Web.Linkname%)
    # This is an old JosWiki render option combined with the new Web.LinkName notation
    # (Uncomment for JosWiki compatibility)
#    s/(^|\s|\()\%([a-zA-Z0-9]+)\.(.*?[^\s])\%(\s|\)|$)/&internalLink($2,$3,$3,$1,1)/geo;

    return $_;
}


# =========================
sub extendGetRenderedVersionInsidePRE
{
    # This is the place to define customized rendering rules
    # Called by sub getRenderedVersion, in loop inside of <PRE> tag

    my( $text, $theWeb ) = @_;

    # do custom extension rule, like for example:
    # s/old/new/go;

    return $_;
}

# ========================= XP support routines RJB 2001.03.14

sub renderHTML {
  my $text = $_[0];
  $text =~ s/\</&lt;/g;
  $text =~ s/\>/&gt;/g;
  return $text;
} 

###########################
# xpShowStories
#
# Displays the complete list of stories.

sub xpShowStories {
    my $theWeb = $_[0];

    my @statusLiterals = ("Waiting", "In progress", "COMPLETE", "Acceptance"); # waiting, progress, complete

    my $list = "<table border=\"1\">";

    my @allStories = xpGetAllStories($theWeb);  
    
    # Iterate over each, and build story hash
    my (%stories,%storyLead) = ();
    foreach my $story (@allStories) {
	my $storyText = &wiki::readTopic($story);
	if ($storyText =~ /<!--storyiter--> *(.*?) *<!--\/storyiter-->/s) {
	    $stories{$story} = $1;
	} else {
	    $stories{$story} = "none";
	}
	if ($storyText =~ /<!--storyLead--> *(.*?) *<!--\/storyLead-->/s) {
	    $storyLead{$story} = $1;
	} else {
	    $storyLead{$story} = "[ none ]";
        }
    }
    
    # Show the list
    my $iteration = "";
    foreach my $story (sort { $stories{$a} cmp $stories{$b} || $a cmp $b } keys %stories) {
	if ($stories{$story} ne $iteration) {
	    $iteration = $stories{$story};
	    $list .= "<tr bgcolor=\"#CCCCCC\"><th align=\"left\">Stories for iteration: ".$iteration." </th><th>Status</th><th>Story lead</th></tr>";
	}
	my $status = xpGetStoryStatus(&wiki::readTopic($story));
	$list .= "<tr><td> ".$story." </td><td>".$statusLiterals[$status]."</td><td> ".$storyLead{$story}." </td></tr>";
    }

    # Close it off
    $list .= "</table>";

}

###########################
# xpDumpIteration
#
# Dumps stories and tasks in an iteration.

sub xpDumpIteration {
    my ($iteration,$theWeb) = @_;

    my @allStories = xpGetAllStories($theWeb);  
    
    # Iterate over each and build master list

    my $bigList = "";

    foreach my $story (@allStories) {
	my $storyText = &wiki::readTopic($story);
	# TODO: This is a hack!
	# Patch the embedded "DumpStoryList" name to the real story name
	if ($storyText =~ /<!--storyiter--> *(.*?) *<!--\/storyiter-->/s) {
	    if ($1 eq $iteration) {
		# TODO: This is a hack!
		# Patch the embedded %TOPIC% before the main TWiki code does
		$storyText =~ s/%TOPIC%/$story/go;
		$bigList .= "<h2>Story: ".$story."</h2>\n".$storyText."<br><br><hr> \n";
	    }
	}
    }
    
    return $bigList;

}

###########################
# xpShowIteration
#
# Shows the specified iteration broken down by stories and tasks

sub xpShowIteration {

    my ($iterationName,$theWeb) = @_;

    my @statusLiterals = ("Waiting", "In progress", "", "Acceptance"); # waiting, progress, complete, acceptance testing

    my $list = "<table border=\"1\">";
    $list .= "<tr bgcolor=\"#CCCCCC\"><th align=\"left\">Story (orig. story estimate) / Tasks </th><th>Estimate<br>(ideals)</th><th>Who</th><th>Spent<br>(ideals)</th><th>To do<br>(ideals)</th><th>Status</th></tr>";
 
    my @allStories = xpGetAllStories($theWeb);  
        
    # Iterate over each, and see if belong to this iteration, if so add to hash
    my (%targetStories,%targetOrder) = ();
    foreach my $story (@allStories) {
	my $storyText = &wiki::readTopic($story);
	if ( $storyText =~ /<!--storyiter--> *$iterationName *<!--\/storyiter-->/s ) {
	    $targetStories{$story} = $storyText;
	    # Get the ordering and save it
            $storyText =~ /<!--order--> *(.*?) *<!--\/order-->/s;
	    $targetOrder{$story} = $1;
	}
    }
    
    my ($totalSpent,$totalEtc,$totalEst,$totalStoryEst) = 0;

    # Show them
    foreach my $story (sort { $targetOrder{$a} <=> $targetOrder{$b} || $a cmp $b } keys %targetStories) {
	my $text = $targetStories{$story};
	
	# Get story estimate
	$text =~ /<!--storyest--> *(.*?) *<!--\/storyest-->/s;
	my $storyEst = $1;

	# Get acceptance test status
	my $storyComplete = "N";
	if ($text =~ /<!--complete--> *(.*?) *<!--\/complete-->/s) {
	    $storyComplete = uc(substr($1,0,1));
	}

	# Get story lead
	my $storyLead = "";
	if ($text =~ /<!--storyLead--> *(.*?) *<!--\/storyLead-->/s) {
	    $storyLead = $1;
	}

	# Set up other story stats
	my ($storySpent,$storyEtc,$storyCalcEst) = 0;
	
	# Suck in the tasks
	my (@taskName, @taskStat, @taskEst, @taskWho, @taskSpent, @taskEtc) = (); # arrays for each task
	my $taskCount = 0; # Amount of tasks in this story
	my @storyStat = (); # Array of counts of task status
	
	while ($text =~ s/<!--taskname--> *(.*?) *?<!--\/taskname-->//s) {
	    $taskName[$taskCount] = $1;
	    $text =~ s/<!--est--> *(.*?) *?<!--\/est-->//s;
	    $taskEst[$taskCount] = $1;
	    $text =~ s/<!--who--> *(.*?) *?<!--\/who-->//s;
	    $taskWho[$taskCount] = $1;
	    $text =~ s/<!--spent--> *(.*?) *?<!--\/spent-->//s;
	    $taskSpent[$taskCount] = $1;
	    $text =~ s/<!--etc--> *(.*?) *?<!--\/etc-->//s;
	    $taskEtc[$taskCount] = $1;
	    
	    # Calculate status of task ; 0=waiting, 1=progress, 2=complete
	    $taskStat[$taskCount] = xpTaskStatus($taskWho[$taskCount],$taskEtc[$taskCount],$taskSpent[$taskCount]);
	    
	    $storyStat[$taskStat[$taskCount]]++;
	    
	    # Calculate spent
	    my @spentList = xpRipWords($taskSpent[$taskCount]);
	    foreach my $spent (@spentList) {
		$storySpent += $spent;
	    }
	    
	    # Calculate etc
	    my @etcList = xpRipWords($taskEtc[$taskCount]);
	    foreach my $etc (@etcList) {
		$storyEtc += $etc;
	    }
	    
	    # Calculate est
	    my @estList = xpRipWords($taskEst[$taskCount]);
	    foreach my $etc (@estList) {
		$storyCalcEst += $etc;
	    }
	    
	    $taskCount++;
	    
	}

	# Calculate story status
	if ( ($storyStat[1] == 0) and ($storyStat[2] == 0) ) { # All tasks are waiting
	    $storyStat = $statusLiterals[0];
	} elsif ( ($storyStat[0] == 0) and ($storyStat[1] == 0) ) { # All tasks complete
	    if ($storyComplete eq "Y") {
		$storyStat = $statusLiterals[2];
	    } else {
		$storyStat = $statusLiterals[3];
	    }
	} else {
	    $storyStat = $statusLiterals[1];
	}
	
	# Show story line
	$list .= "<tr";
	if ($storyComplete eq "N") {
	    $list .= " bgcolor=\"#99FF99\"";
	}
	$list .= "><td> ".$story." (".$storyEst.")</td><td align=\"center\"><b>".$storyCalcEst."</b></td><td> ".$storyLead." </td><td align=\"center\"><b>".$storySpent."</b></td><td align=\"center\">";
	if ($storyEtc != 0) {
	    $list .= "<b>".$storyEtc."</b>";
	}
	$list .= "</td><td nowrap>".$storyStat."</td></tr>";
	
	# Show each task
	for (my $i=0; $i<$taskCount; $i++) {

	    my $taskBG = "";
	    if ($taskStat[$i] != 2) {
		$taskBG = " bgcolor=\"#FFCCCC\"";
	    }

	    # Line for each engineer
	    my $doName = 1;
	    my @who = xpRipWords($taskWho[$i]);
	    my @est = xpRipWords($taskEst[$i]);
	    my @spent = xpRipWords($taskSpent[$i]);
	    my @etc = xpRipWords($taskEtc[$i]);
	    for (my $x=0; $x<@who; $x++) {
		$list .= "<tr".$taskBG."><td>&nbsp;";
		if ($doName) {
		    $list .= "&nbsp;&nbsp;&nbsp;".$taskName[$i];
		}
		$list .= "</td><td align=\"center\">".xpZero2Null($est[$x])."</td><td> ".$who[$x]." </td><td align=\"center\">".$spent[$x]."</td><td align=\"center\">".xpZero2Null($etc[$x])."</td><td nowrap>";

		$list .= $statusLiterals[$taskStat[$i]];

		$list .= "</td></tr>";
		$doName = 0;
	    }

	}

	# Add a spacer
	$list .= "<tr><td colspan=\"6\">&nbsp;</td></tr>";

	# Add to totals
	$totalSpent += $storySpent;
	$totalEtc += $storyEtc;
	$totalEst += $storyCalcEst;
	$totalStoryEst += $storyEst;
	
    }

    # Do iteration totals

    $list .= "<tr bgcolor=\"#CCCCCC\"><td><b>Team totals</b></td><td align=\"center\"><b>".$totalEst."</b></td><td></td><td align=\"center\"><b>".$totalSpent."</b></td><td align=\"center\"><b>".$totalEtc."</b></td><td></td></tr>";
    
    $list .= "</table>";
    
    return $list;
}


###########################
# xpShowProjectStories
#
# Shows the project completion by release and iteration using stories.

sub xpShowProjectStories {
    my $theWeb = $_[0];

    my @allStories = xpGetAllStories($theWeb);

    # Show the list
    my $list = "<table border=\"1\"><tr bgcolor=\"#CCCCCC\"><th>Iteration</th><th>Waiting</th><th>In progress</th><th>Task complete</th><th colspan=\"2\">Complete</th><th>Total</th></tr>";

    # Iterate over each, and build iteration hash
    my ($waiting,$progress,$tcomplete,$complete,$total) = 0;
    my (%master,%waiting,%progress,%tcomplete,%complete) = ();
    foreach my $story (@allStories) {
	my $storyText = &wiki::readTopic($story);
	$storyText =~ /<!--storyiter--> *(.*?) *<!--\/storyiter-->/s;
	my $iter = $1;
	if ($iter ne "TornUp") {
	    $master{$iter}++;
	    my $status = xpGetStoryStatus($storyText);
	    if ($status == 0) {
		$waiting{$iter}++;
		$waiting++;
	    } elsif ($status == 1) {
		$progress{$iter}++;
		$progress++;
	    } elsif ($status == 3) {
		$tcomplete{$iter}++;
		$tcomplete++;
	    } else {
		$complete{$iter}++;
		$complete++;
	    }
	    $total++;
	}
    }

    # Get date of each iteration
    my %iterKeys = ();
    foreach my $iteration (keys %master) {
        my $iterText = &wiki::readTopic($iteration);
        $iterText =~ /\<!--START *(.*?) *--\>/s;
        $iterKeys{$iteration} = $1;
    }

    # OK, display them
    foreach my $iteration (sort { $iterKeys{$a} <=> $iterKeys{$b} } keys %master) {
	my $pctComplete = 0;
	if ($complete{$iteration} > 0) {
	    $pctComplete = sprintf("%u",($complete{$iteration}/$master{$iteration})*100);
	}
	$list .= "<tr><td> ".$iteration." </td><td align=\"center\">".$waiting{$iteration}."</td><td align=\"center\">".$progress{$iteration}."</td><td align=\"center\">".$tcomplete{$iteration}."</td><td align=\"center\">".$complete{$iteration}."</td><td align=\"center\">".$pctComplete."\%<td align=\"center\">".$master{$iteration}."</td></tr>";
    }
    my $pctComplete = 0;
    if ($complete > 0) {
	$pctComplete = sprintf("%u",($complete/$total)*100);
    }
    $list .= "<tr bgcolor=\"#CCCCCC\"><th align=\"left\">Totals</th><th>".$waiting."</th><th>".$progress."</th><th>".$tcomplete."</th><th>".$complete."</th><th>".$pctComplete."%</th><th>".$total."</th></tr>";

    # Close it off
    $list .= "</table>";

    return $list;
}

###########################
# xpShowProjectTasks
#
# Shows the project completion by release and iteration using tasks.

sub xpShowProjectTasks {
    my $theWeb = $_[0];

    my @allStories = xpGetAllStories($theWeb);

    # Show the list
    my $list = "<table border=\"1\"><tr bgcolor=\"#CCCCCC\"><th>Iteration</th><th>Waiting</th><th>In progress</th><th colspan=\"2\">Complete</th><th>Total</th></tr>";

    # Iterate over each, and build iteration hash
    my ($waiting,$progress,$complete,$total) = 0;
    my (%master,%waiting,%progress,%complete) = ();
    foreach my $story (@allStories) {
	my $storyText = &wiki::readTopic($story);
	$storyText =~ /<!--storyiter--> *(.*?) *<!--\/storyiter-->/s;
	my $iter = $1;
	if ($iter ne "TornUp") {
	    while (1) {
		(my $status,$storyText,my $taskName,my $taskEst,my $taskWho,my $taskSpent,my $taskEtc,my $taskStatus) = xpGetNextTask($storyText);
		if (!$status) {
		    last;
		}
		$master{$iter}++;
		if ($taskStatus == 0) {
		    $waiting{$iter}++;
		    $waiting++;
		} elsif ( ($taskStatus == 1) or ($taskStatus == 3) ) {
		    $progress{$iter}++;
		    $progress++;
		} else {
		    $complete{$iter}++;
		    $complete++;
		}
		$total++;
	    }
	}
    }

    # Get date of each iteration
    my %iterKeys = ();
    foreach my $iteration (keys %master) {
        my $iterText = &wiki::readTopic($iteration);
        $iterText =~ /\<!--START *(.*?) *--\>/s;
        $iterKeys{$iteration} = $1;
    }

    # OK, display them
    foreach my $iteration (sort { $iterKeys{$a} <=> $iterKeys{$b} } keys %master) {
	my $pctComplete = 0;
	if ($complete{$iteration} > 0) {
	    $pctComplete = sprintf("%u",($complete{$iteration}/$master{$iteration})*100);
	}
	$list .= "<tr><td> ".$iteration." </td><td align=\"center\">".$waiting{$iteration}."</td><td align=\"center\">".$progress{$iteration}."</td><td align=\"center\">".$complete{$iteration}."</td><td align=\"center\">".$pctComplete."\%<td align=\"center\">".$master{$iteration}."</td></tr>";
    }
    my $pctComplete = 0;
    if ($complete > 0) {
	$pctComplete = sprintf("%u",($complete/$total)*100);
    }
    $list .= "<tr bgcolor=\"#CCCCCC\"><th align=\"left\">Totals</th><th>".$waiting."</th><th>".$progress."</th><th>".$complete."</th><th>".$pctComplete."%</th><th>".$total."</th></tr>";

    # Close it off
    $list .= "</table>";

    return $list;
}

###########################
# xpTest
#
# Implements the %XPTEST% variable. Use this for anything you need to test.

sub xpTest {

}

###########################
# xpTaskStatus
#
# Calculates the status of a task.

sub xpTaskStatus {
    my @who = xpRipWords($_[0]);
    my @etc = xpRipWords($_[1]);
    my @spent = xpRipWords($_[2]);

    # status - 0=waiting, 1=inprogress, 2=complete

    # anyone assigned?
    if (@who == 0) {
	return 0; # nobody assigned, waiting
    }
    foreach my $who (@who) {
	if ($who eq "?") {
	    return 0; # not assigned correctly, waiting
	}
    }

    # someone is assigned, see if ANY time remaining
    my $isRemaining = 0;
    foreach my $etc (@etc) {
	if ($etc eq "?") {
	    return 0; # no "todo", so still waiting
	}
	if ($etc > 0) {
	    $isRemaining = 1;
	}
    }
    if (!$isRemaining) {
	return 2; # If no time remaining, must be complete
    }

    # If ANY spent > 0, then in progress, else waiting
    foreach my $spent (@spent) {
	if ($spent > 0) {
	    return 1; # in progress
	}
    }
    return 0;

}

###########################
# xpShowVelocities
#
# Shows velocities of resources in an iteration.

sub xpShowVelocities {
    my ($iteration,$theWeb) = @_;

    my @allStories = xpGetAllStories($theWeb);

    # Show the list
    my $list = "<table border=\"1\"><tr bgcolor=\"#CCCCCC\"><th rowspan=\"2\">Who</th><th colspan=\"3\">Ideals</th><th colspan=\"2\">Tasks</th></tr><tr bgcolor=\"#CCCCCC\"><th>Assigned</th><th>Spent</th><th>Remaining</th><th>Assigned</th><th>Remaining</th></tr>";

    # Iterate over each story
    my (%whoAssigned,%whoSpent,%whoEtc,%whoTAssigned,%whoTRemaining) = ();
    my ($totalSpent,$totalEtc,$totalAssigned,$totalVelocity,$totalTAssigned) = 0;
    foreach my $story (@allStories) {
	my $storyText = &wiki::readTopic($story);
	$storyText =~ /<!--storyiter--> *(.*?) *<!--\/storyiter-->/s;
	if ($1 eq $iteration) {
	    while (1) {
		(my $status,$storyText,my $taskName,my $taskEst,my $taskWho,my $taskSpent,my $taskEtc,my $taskStatus) = xpGetNextTask($storyText);
		if (!$status) {
		    last;
		}
		my @who = xpRipWords($taskWho);
		my @spent = xpRipWords($taskSpent);
		my @est = xpRipWords($taskEst);
		my @etc = xpRipWords($taskEtc);
		for (my $i=0; $i<@who; $i++) {
		    $whoSpent{$who[$i]} += $spent[$i];
		    $totalSpent += $spent[$i];

		    $whoEtc{$who[$i]} += $etc[$i];
		    $totalEtc += $etc[$i];

		    $whoAssigned{$who[$i]} += $est[$i];
		    $totalAssigned += $est[$i];

		    $whoTAssigned{$who[$i]}++;
		    $totalTAssigned++;

		    if ($etc[$i] > 0) {
			$whoTRemaining{$who[$i]}++;
		        $totalTRemaining++;
		    }
		}
	    }
	}
    }
    
    # Show them
    foreach my $who (sort { $whoEtc{$b} <=> $whoEtc{$a} } keys %whoSpent) {
	$list .= "<tr><td> ".$who." </td><td align=\"center\">".$whoAssigned{$who}."</td><td align=\"center\">".$whoSpent{$who}."</td><td align=\"center\">".xpZero2Null($whoEtc{$who})."</td><td align=\"center\">".$whoTAssigned{$who}."</td><td align=\"center\">".$whoTRemaining{$who}."</td></tr>";
    }
    $list .= "<tr bgcolor=\"#CCCCCC\"><th align=\"left\">Total</th><th>".$totalAssigned."</th><th>".$totalSpent."</th><th>".$totalEtc."</th><th>".$totalTAssigned."</th><th>".$totalTRemaining."</th></tr>";

    # Close it off
    $list .= "</table>";

    return $list;
}
###########################
# xpGetAllStories
#
# Returns a list of all stories in this web.

sub xpGetAllStories {
    my $theWeb = $_[0];

    # Read in all stories in this web
    opendir(WEB,$dataDir."/".$theWeb);
    my @allStories = grep { s/(.*?Story).txt$/$1/go } readdir(WEB);
    closedir(WEB);
    
    return @allStories;
}

###########################
# xpGetStoryStatus
#
# Returns the status of a story

sub xpGetStoryStatus {
    my $storyText = $_[0];

    my @storyStatus = ();

    # Get acceptance test status
    my $storyComplete = "N";
    if ($storyText =~ /<!--complete--> *(.*?) *<!--\/complete-->/s) {
	$storyComplete = uc(substr($1,0,1));
    }
    
    # Run through tasks and get their status
    while (1) {
	(my $status,$storyText,my $taskName,my $taskEst,my $taskWho,my $taskSpent,my $taskEtc,my $taskStatus) = xpGetNextTask($storyText);
	if (!$status) {
	    last;
	}
	$storyStatus[$taskStatus]++;
    }

    # Calculate story status
    if ( ($storyStatus[1] == 0) and ($storyStatus[2] == 0) ) { # All tasks are waiting
	$storyStatus = 0;
    } elsif ( ($storyStatus[0] == 0) and ($storyStatus[1] == 0) ) { # All tasks complete
	if ($storyComplete eq "Y") {
	    $storyStatus = 2;
	} else {
	    $storyStatus = 3;
	}
    } else {
	$storyStatus = 1;
    }
    
    return $storyStatus;
}    
    
###########################
# xpRipWords
#
# Parses a bunch of words from TWiki code

sub xpRipWords {
    my $string = $_[0];
    my @out = ();
    foreach my $word (split(/[ \|]/,$string)) {
	if ($word ne "") {
	    push @out,$word;
	}
    }
    return @out;
}

###########################
# xpZero2Null
#
# Returns a numeric, or null if zero

sub xpZero2Null {
    if ($_[0] == 0) {
	return "";
    } else {
        return $_[0];
    }
}

###########################
# xpGetNextTask
#
# Return the next task in a story

sub xpGetNextTask {
    my $storyText = $_[0];

    my $taskName;
    if ($storyText =~ s/<!--taskname--> *(.*?) *<!--\/taskname-->//s) {
	$taskName = $1;
    } else {
	return 0;
    }

    $storyText =~ s/<!--est--> *(.*?) *<!--\/est-->//s;
    my $taskEst = $1;

    $storyText =~ s/<!--who--> *(.*?) *<!--\/who-->//s;
    my $taskWho = $1;

    $storyText =~ s/<!--spent--> *(.*?) *<!--\/spent-->//s;
    my $taskSpent = $1;

    $storyText =~ s/<!--etc--> *(.*?) *<!--\/etc-->//s;
    my $taskEtc = $1;
    
    # Calculate status of task ; 0=waiting, 1=progress, 2=complete
    my $taskStatus = xpTaskStatus($taskWho,$taskEtc,$taskSpent);

    return (1,$storyText,$taskName,$taskEst,$taskWho,$taskSpent,$taskEtc,$taskStatus);
}
    
